//
//  PredictionCache.swift
//  death_app Watch App
//
//  Caching system for prediction engine performance optimization
//

import Foundation

class PredictionCache {
    private let cache = NSCache<NSString, CachedPrediction>()
    private let userDefaults = UserDefaults.standard
    private let lastPredictionKey = "lastLifeExpectancyPrediction"
    
    init() {
        cache.countLimit = 50 // Limit cache size
        cache.totalCostLimit = 1024 * 1024 // 1MB limit
    }
    
    /// Store prediction in cache
    func storePrediction(_ prediction: LifeExpectancyPrediction, for key: String) {
        let cached = CachedPrediction(prediction: prediction, timestamp: Date())
        cache.setObject(cached, forKey: NSString(string: key))
        
        // Also store as last prediction for change calculations
        storeLastPrediction(prediction)
    }
    
    /// Get cached prediction if still valid
    func getPrediction(for key: String) -> LifeExpectancyPrediction? {
        guard let cached = cache.object(forKey: NSString(string: key)) else {
            return nil
        }
        
        // Check if cache is still valid (24 hours)
        let cacheValidityPeriod: TimeInterval = 24 * 60 * 60
        guard Date().timeIntervalSince(cached.timestamp) < cacheValidityPeriod else {
            cache.removeObject(forKey: NSString(string: key))
            return nil
        }
        
        return cached.prediction
    }
    
    /// Get the last stored prediction for change calculations
    func getLastPrediction() -> LifeExpectancyPrediction? {
        guard let data = userDefaults.data(forKey: lastPredictionKey),
              let prediction = try? JSONDecoder().decode(LifeExpectancyPrediction.self, from: data) else {
            return nil
        }
        return prediction
    }
    
    /// Store the last prediction persistently
    private func storeLastPrediction(_ prediction: LifeExpectancyPrediction) {
        if let data = try? JSONEncoder().encode(prediction) {
            userDefaults.set(data, forKey: lastPredictionKey)
        }
    }
    
    /// Clear all cached predictions
    func clearCache() {
        cache.removeAllObjects()
    }
    
    /// Invalidate cache based on data changes
    func invalidateCache(for dataType: CacheInvalidationType) {
        // In a more sophisticated implementation, would selectively invalidate
        // For now, clear all cache when any health data changes
        clearCache()
    }
}

// MARK: - Supporting Types

private class CachedPrediction {
    let prediction: LifeExpectancyPrediction
    let timestamp: Date
    
    init(prediction: LifeExpectancyPrediction, timestamp: Date) {
        self.prediction = prediction
        self.timestamp = timestamp
    }
}

enum CacheInvalidationType {
    case healthData
    case surveyData
    case demographics
    case all
}